/* -*-  Mode: C++; c-file-style: "gnu"; indent-tabs-mode:nil; -*- */
/*
 * Copyright (c) 2005 INRIA
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as 
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Mathieu Lacage <mathieu.lacage@sophia.inria.fr>
 */

#include "mac-high-adhoc.h"
#include "dca-txop.h"
#include "wifi-net-device.h"
#include "mac-stations.h"
#include "wifi-phy.h"
#include "ns3/packet.h"
#include "ns3/log.h"

NS_LOG_COMPONENT_DEFINE ("MacHighAdhoc");

namespace ns3 {

MacHighAdhoc::MacHighAdhoc ()
{}
MacHighAdhoc::~MacHighAdhoc ()
{
  m_phy = 0;
}

void
MacHighAdhoc::SetDevice (WifiNetDevice *device)
{
  m_device = device;

}
void 
MacHighAdhoc::SetForwardCallback (ForwardCallback callback)
{
  m_callback = callback;
}
void
MacHighAdhoc::SetDcaTxop (DcaTxop *dca)
{
  m_dca = dca;
}
void 
MacHighAdhoc::SetStations (MacStations *stations)
{
  m_stations = stations;
}
void 
MacHighAdhoc::SetPhy (Ptr<WifiPhy> phy)
{
  m_phy = phy;
}

Mac48Address 
MacHighAdhoc::GetBssid (void) const
{
  // XXX the bssid should be generated by the procedure
  // described in ieee802.11 section 11.1.3
  return Mac48Address::GetBroadcast ();
}

void 
MacHighAdhoc::Enqueue (Ptr<const Packet> packet, Mac48Address to)
{
  NS_LOG_DEBUG ("enqueue size="<<packet->GetSize ()<<", to="<<to);
  WifiMacHeader hdr;
  hdr.SetType (WIFI_MAC_DATA);
  hdr.SetAddr1 (to);
  hdr.SetAddr2 (m_device->GetSelfAddress ());
  hdr.SetAddr3 (m_device->GetBssid ());
  hdr.SetDsNotFrom ();
  hdr.SetDsNotTo ();

  MacStation *destination = m_stations->Lookup (to);
  if (destination->IsBrandNew ())
    {
      // in adhoc mode, we assume that every destination
      // supports all the rates we support.
      for (uint32_t i = 0; i < m_phy->GetNModes (); i++)
        {
          destination->AddSupportedMode (m_phy->GetMode (i));
        }
      destination->RecordDisassociated ();
    }

  m_dca->Queue (packet, hdr);
}

void 
MacHighAdhoc::Receive (Ptr<Packet> packet, WifiMacHeader const *hdr)
{
  NS_LOG_DEBUG ("received size="<<packet->GetSize ()<<", from="<<hdr->GetAddr2 ());
  if (hdr->GetAddr1 ().IsBroadcast () || hdr->GetAddr1 () == m_device->GetSelfAddress ())
    {
      m_callback (packet, hdr->GetAddr2 ());
    }
}

} // namespace ns3
